<?php
/* --------------------------------------------------------------
 AbstractFooterBadgeLoader.php 2020-04-16
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

namespace Gambio\Admin\Layout\Smarty\Loaders;

use Smarty;

/**
 * Class AbstractFooterBadgeLoader
 * @package Gambio\Admin\Layout\Smarty\Loaders\FooterBadge
 */
abstract class AbstractFooterBadgeLoader implements LoaderInterface
{
    public function load(Smarty $smarty): void
    {
        $script = $this->renderScriptTag();
        $this->safeAssign('dynamic_script_assets', $script, $smarty);
    }
    
    
    /**
     * Renders the script tag.
     *
     * @return string
     */
    private function renderScriptTag(): string
    {
        $isConnected   = $this->isConnected();
        $dataConnected = $isConnected ? 'data-connected' : '';
        
        return <<<SCRIPT
<script src="{$this->jsExtenderSrc()}"
data-text="{$this->dataText($isConnected)}" {$dataConnected}
></script>
SCRIPT;
    }
    
    
    /**
     * Safely assigns the key and value to smarty.
     *
     * Safely means that if the key is already used, the value will be appended
     * instead of replaced.
     *
     * @param string $key
     * @param string $value
     * @param Smarty $smarty
     */
    private function safeAssign(string $key, string $value, Smarty $smarty): void
    {
        $templateValue = $smarty->getTemplateVars($key);
        
        if (!$templateValue) {
            $smarty->assign($key, $value);
            
            return;
        }
        
        if (is_string($templateValue)) {
            $newValue = $templateValue . PHP_EOL . $value;
            $smarty->assign($key, $newValue);
            
            return;
        }
        
        if (is_array($templateValue)) {
            $templateValue[] = $value;
            $smarty->assign($key, $templateValue);
        }
    }
    
    
    /**
     * Finally determines the color of the badge.
     *
     * @return bool
     */
    abstract protected function isConnected(): bool;
    
    
    /**
     * Evaluates the src path to the js extender file.
     *
     * @return string
     */
    abstract protected function jsExtenderSrc(): string;
    
    
    /**
     * Provides the data-text attribute for the script tag.
     *
     * @param bool $isConnected
     *
     * @return string
     */
    abstract protected function dataText(bool $isConnected): string;
}